<?php
/*
Plugin Name: Similar Posts 
Plugin URI: http://rmarsh.com/2006/10/04/similar-posts/
Description: Displays a list of the most similar posts to the current one by comparing the pattern of word usage.
Version: 1.14
Author: Rob Marsh, SJ
Author URI: http://rmarsh.com/
*/

/*
Copyright 2006  Rob Marsh, SJ  (http://rmarsh.com)

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

global $spo_index_exists, $overusedwords;

/*
	Template Tag: Displays similar posts according to pattern of word usage.
	*************************************************************
	Various options can be set on the Similar Posts options page or passed to the
	template tag as a WordPress-style query string, e.g.:
	
		<?php similar_posts('limit=10') ?> 
			lists the 10 most similar posts
		<?php similar_posts('none_text=nothing found&show_static=false') ?> 
			lists the default number of posts, excluding static pages, and specifies
			what to display if there are none
		
	If you do not specify an option its default value is used according to the 
	values set on the option page.
	
	This means you can use the template tag in different ways in different places.
	
	The full list of parameters is as follows:
	
		limit			maximum number of posts to show (5)
		skip			how many posts to skip before listing (0)
		show_static		include static pages (false)
		show_private	include password-protected posts (false)
		excluded_cats	comma separated list of categories to exclude (by ID) (9999, the default means none)
		excluded_authors	comma separated list of authors to exclude (by ID) (9999, the default means none)
		none_text		what to show if no posts match--can be plain text or a permalink
		before_title	what to show before a link (<li>)
		after_title		what to show after a link(</li>)
		trim_before	remove the first instance of 'before_title' (false)
		show_excerpt	include a snippet of the post after the link (false)
		excerpt_length	max number of characters in excerpt (50)
		excerpt_format	'char', the default, does nothing, 'word' trims the excerpt to the last full word, and 'sent' to the full sentence. 
					If the excerpt would be trimmed to nothing no trimming is applied.
		ellipsis		add ' ...' after the excerpt			
		before_excerpt	what to show before an excerpt (<li>)
		after_excerpt	what to show after an excerpt (</li>)
		
		
*/

function similar_posts($args = '') {
	global $wpdb, $post, $spo_index_exists;
	if (!$spo_index_exists) {
		echo $before_title.__('Similar Posts fulltext index is missing').$after_title;
		return;
	}
	if (!function_exists('process_input')) require_once 'helpers.php';
	$p = process_input($args);
	
	// Make sure the post is not from the future
	$time_difference = get_settings('gmt_offset');
	$now = gmdate("Y-m-d H:i:s",(time()+($time_difference*3600)));
	//set up a bit of the query string
	$showstatus = ($p['show_static']=='true') ? "'publish', 'static'" : "'publish'";
	$showpass = ($p['show_private']=='false') ? "AND post_password ='' " : "";
	$excluded_cats = $p['excluded_cats'];
	$excluded_authors = $p['excluded_authors'];
	$limit = $p['skip'].', '.$p['limit'];
	
	//the workhorse...
	//get the terms to do the matching
	//see if there's a custom field stored for this post
	$postcustom = get_post_custom_values('similarterms');
	if (empty($postcustom)) {
		//if not generate the terms on the fly
		$terms = get_terms($post->post_content . ' ' . $post->post_title);
	} else {
		//use the stored terms
		$terms = addslashes($postcustom[0]);
	}
	// Primary SQL query against the full text index 
    $sql = "SELECT DISTINCT ID, post_title, post_content "
		. "FROM $wpdb->posts LEFT JOIN $wpdb->post2cat ON ID = post_id "
		. "WHERE MATCH (post_title, post_content) AGAINST ('$terms' )"
		. "AND post_date <= '$now' "
		. "AND (post_status IN ( $showstatus ) && ID != '$post->ID' && category_id NOT IN ( $excluded_cats ) && post_author NOT IN ( $excluded_authors ) ) "
		. $showpass . "LIMIT $limit";
    $results = $wpdb->get_results($sql);

	echo process_output($results, $p);
	
}

/*******************************************************************************************************
	The functions beyond this point are for internal use only
********************************************************************************************************/

/*
	Takes a string, strips it of html, and produces list of the 20 most used words (common words ignored)
*/
function get_terms($string) {
	global $overusedwords;
	//tidy up the string a little 
	$string = strtolower(trim(addslashes(strip_tags($string))));
	//make an array of all the words in the string (because we have used addslashes above this will chop off apostrophised endings)
	$wordlist = str_word_count($string, 1);
	//count them, word=>count
	$wordtable = array_count_values($wordlist);
	//knock out the noise words ... the $overusedwords array is loaded from a file to allow for different languages
	foreach ($overusedwords as $word) {
		unset($wordtable[$word]); 
	}
	// knock out words of three or less characters sicne mysql ignores them for full text searches
	foreach ($wordtable as $word => $freq) {
		if (strlen($word) < 4) {
			unset($wordtable[$word]);
		}
	}
	//sort by count
	arsort($wordtable);
	//convert the first 20 words (the most used) into a list 'term1 term2 term3' etc.
	$terms = implode(' ', array_slice(array_keys($wordtable), 0, 20));
	return $terms;
}

/*
	Given the ID of a post, gets its most common words and stores them in a custom field called 'similarterms'
*/
function save_similar_terms($postID) {
	global $wpdb;
	//get the post content and title
	$content = $wpdb->get_row("SELECT post_content, post_title FROM $wpdb->posts WHERE ID = $postID", ARRAY_A);
	//extract its terms
	$terms = get_terms($content['post_content'] . ' ' . $content['post_title']);
	//check to see if the field is set
	$metaid = $wpdb->get_var("SELECT meta_id FROM $wpdb->postmeta WHERE post_id=$postID AND meta_key='similarterms' limit 1");
	//then either insert or update accordingly
	if (is_null($metaid)) {
		$wpdb->query("INSERT INTO $wpdb->postmeta (post_id, meta_key, meta_value) VALUES ($postID, 'similarterms', \"$terms\")");
	} 
	return $postID;
}

/*
	Given the ID of a post delete its custom field, 'similarterms'
*/
function delete_similar_terms($postID) {
	global $wpdb;
	//check to see if that post has its custom field set
	$metaid = $wpdb->get_var("SELECT meta_id FROM $wpdb->postmeta WHERE post_id=$postID AND meta_key='similarterms' limit 1");
	if (!is_null($metaid)) {
		//then delete the custom field
		$wpdb->query("DELETE FROM $wpdb->postmeta WHERE post_id = $postID AND meta_key = 'similarterms'");
	}
	return $postID;
}

/*
	Work through every post and make sure it has its custom field set
*/
function save_similar_terms_for_all_posts () {
	global $wpdb;
	$termcount = 0;
	//get every post's ID
	$postIDs = $wpdb->get_results("SELECT ID FROM $wpdb->posts", ARRAY_A);
	if (!is_null($postIDs)) {
		//for each post
		foreach ($postIDs as $pID) {
			$postID = $pID['ID'];
			//check if it already has its custom field
			$metaid = $wpdb->get_var("SELECT meta_id FROM $wpdb->postmeta WHERE post_id=$postID AND meta_key='similarterms' limit 1");
			if (is_null($metaid)) {
				//if not process the content for similar terms
				$content = $wpdb->get_row("SELECT post_content, post_title FROM $wpdb->posts WHERE ID = $postID", ARRAY_A);
				$terms = get_terms($content['post_content'] . ' ' . $content['post_title']);
				//and store them
				$wpdb->query("INSERT INTO $wpdb->postmeta (post_id, meta_key, meta_value) VALUES ($postID, 'similarterms', \"$terms\")");
				$termcount = $termcount + 1;
			}
		}
	}
	return $termcount;
}

/*
	Remove ALL similarterms custom fields
*/
function clear_similar_terms_for_all_posts () {
	global $wpdb;
	$numdeleted = $wpdb->query("DELETE FROM $wpdb->postmeta WHERE meta_key = 'similarterms'");
	if (!$numdeleted) {
		$numdeleted = 0;
	}
	return $numdeleted;
}

/*
	Import all Related Posts 'keywords' into 'similarterms'
*/
function import_related_posts_keywords () {
	global $wpdb;
	$postcount = 0;
	$keycount = 0;
	//get every post's ID
	$postIDs = $wpdb->get_results("SELECT ID FROM $wpdb->posts", ARRAY_A);
	if (!is_null($postIDs)) {
		//for each post
		foreach ($postIDs as $pID) {
			$postID = $pID['ID'];
			//get the custom 'keyword' field if any
			$keywords = $wpdb->get_results("SELECT meta_value FROM $wpdb->postmeta WHERE post_id=$postID AND meta_key='keyword'", ARRAY_A);
			if (!empty($keywords)) {
				$postcount = $postcount + 1;
				$keycount = $keycount + count($keywords);
				$terms = '';
				foreach ($keywords as $keywordinfo) {
					$terms = $terms . $keywordinfo['meta_value']. ' ';
				}
				$terms = trim(addslashes($terms));
				//check if it already has its custom field
				$metaid = $wpdb->get_var("SELECT meta_id FROM $wpdb->postmeta WHERE post_id=$postID AND meta_key='similarterms' limit 1");
				if (is_null($metaid)) {
					$wpdb->query("INSERT INTO $wpdb->postmeta (post_id, meta_key, meta_value) VALUES ($postID, 'similarterms', \"$terms\")");
				} else {
					$wpdb->query("UPDATE $wpdb->postmeta SET meta_value = \"$terms\" WHERE post_ID = $postID AND meta_key = 'similarterms'");
				}
			}
		}
	}
	$counter = array('posts' => $postcount, 'keywords' => $keycount);
	return $counter;
}

/*
	This sprawling function handles the display and processing of the options page. The options
	apply to all the template tags above. There is also a section specific to similar-posts.
*/
function similar_posts_options_page() {
	global $wpdb, $table_prefix;
	//we begin with four 'if' blocks to handle each of the buttons which might have been pressed on the options page
	//first, we handle the saving of options
	if (isset($_POST['update_options'])) {
		$option_limit = $_POST['spo_limit'];
		$option_skip = $_POST['spo_skip'];
		$option_excerpt_length = $_POST['spo_excerpt_length'];
		$option_before_title = $_POST['spo_before_title'];
		$option_after_title = $_POST['spo_after_title'];
		$option_trim_before = $_POST['spo_trim_before'];
		$option_before_excerpt = $_POST['spo_before_excerpt'];
		$option_after_excerpt = $_POST['spo_after_excerpt'];
		$option_show_private = $_POST['spo_show_private'];
		$option_show_excerpt = $_POST['spo_show_excerpt'];
		$option_show_static = $_POST['spo_show_static'];
		$option_none_text = $_POST['spo_none_text'];
		// the posted variable is an array but we want to store it as a comma separated list of values ... '9999' === no categories
		if ($_POST['spo_excluded_cats']) {
			$option_excluded_cats = rtrim(implode(',', $_POST['spo_excluded_cats']), ',');
		} else {
			$option_excluded_cats = '9999';
		}
		// the posted variable is an array but we want to store it as a comma separated list of values ... '9999' === no categories
		if ($_POST['spo_excluded_authors']) {
			$option_excluded_authors = rtrim(implode(',', $_POST['spo_excluded_authors']), ',');
		} else {
			$option_excluded_authors = '9999';
		}
		$option_excerpt_format = $_POST['spo_excerpt_format'];
		$option_ellipsis = $_POST['spo_ellipsis'];

		update_option('spo_limit', $option_limit);
		update_option('spo_skip', $option_skip);
		update_option('spo_excerpt_length', $option_excerpt_length);
		update_option('spo_before_title', $option_before_title);
		update_option('spo_after_title', $option_after_title);
		update_option('spo_trim_before', $option_trim_before);
		update_option('spo_before_excerpt', $option_before_excerpt);
		update_option('spo_after_excerpt', $option_after_excerpt);
		update_option('spo_show_private', $option_show_private);
		update_option('spo_show_excerpt', $option_show_excerpt);
		update_option('spo_show_static', $option_show_static);
		update_option('spo_none_text', $option_none_text);
		update_option('spo_excluded_cats', $option_excluded_cats);
		update_option('spo_excluded_authors', $option_excluded_authors);
		update_option('spo_excerpt_format', $option_excerpt_format);
		update_option('spo_ellipsis', $option_ellipsis);
		//show a message
		echo '<div class="updated"><p>' . __('Options saved') . '</p></div>';
	}
	 //Second, we handle the bulk processing of similar terms
	if (isset($_POST['process_all'])) {
		//make sure we do this just the once per press
		unset($_POST['process_all']);
		//do the dirty work
		$termcount = save_similar_terms_for_all_posts ();
		//show a message
		echo '<div class="updated"><p>' . __('Saved terms for ') . $termcount . __(' posts.') . '</p></div>';
     }
	 //Third, we handle the clearing of all similar terms
	if (isset($_POST['clear_all'])) {
		//make sure we do this just the once per press
		unset($_POST['clear_all']);
		//do the dirty work
		$numkeys = clear_similar_terms_for_all_posts ();
		//show a message
		echo '<div class="updated"><p>' . __('Cleared ') . $numkeys . __(' keys') .'</p></div>';
     }
	 //Fourth, we handle the import of keywords from the Related Posts plugin
	if (isset($_POST['import_all'])) {
		//make sure we do this just the once per press
		unset($_POST['import_all']);
		//do the dirty work
		$counter = import_related_posts_keywords ();
		//show a message
		echo '<div class="updated"><p>' . __('Imported ') . $counter['keywords'] . __(' keywords from ') . $counter['posts'] . __(' posts.') .'</p></div>';
     }
	
	//now we drop into html to display the other option page forms	
	?>
		<div class="wrap">
		<h2><?php _e('Posts Options') ?></h2>
		<form method="post">
		<fieldset class="options">
		<table>
			<tr>
				<td><label for="spo_limit"><?php _e('How many posts would you like to show?') ?></label>:</td>
				<td><input name="spo_limit" type="text" id="spo_limit" value="<?php echo get_option('spo_limit'); ?>" size="2" /></td>
			</tr>
			<tr>
				<td><label for="spo_skip"><?php _e('How many posts would you like to skip before listing?') ?></label>:</td>
				<td><input name="spo_skip" type="text" id="spo_skip" value="<?php echo get_option('spo_skip'); ?>" size="2" /></td>
			</tr>
		 	<tr>
           		<td><label for="spo_before_title"><?php _e('Before') ?></label> / <label for="spo_after_title"><?php _e('After (Post Title)') ?> </label>:</td>
				<td><input name="spo_before_title" type="text" id="spo_before_title" value="<?php echo htmlspecialchars(stripslashes(get_option('spo_before_title'))); ?>" size="10" /> / <input name="spo_after_title" type="text" id="spo_after_title" value="<?php echo htmlspecialchars(stripslashes(get_option('spo_after_title'))); ?>" size="10" /><em><small> <?php _e('For example: ') ?>&lt;li&gt;&lt;/li&gt; <?php _e('or') ?> &lt;dt&gt;&lt;/dt&gt;</small></em>
				</td>
			</tr>
			<tr>
				<td><?php _e('Remove the first before-post code? This lets you do things like have a comma-separated list.') ?></td>
          		<td>
        			<select name="spo_trim_before" id="spo_trim_before">
        	  		<option <?php if(get_option('spo_trim_before') == 'false') { echo 'selected'; } ?> value="false">False</option>
					<option <?php if(get_option('spo_trim_before') == 'true') { echo 'selected'; } ?> value="true">True</option>
					</select>
				</td> 
			</tr>
			<tr>
				<td><?php _e('Show excerpt?') ?></td>
          		<td>
        			<select name="spo_show_excerpt" id="spo_show_excerpt">
        	  		<option <?php if(get_option('spo_show_excerpt') == 'false') { echo 'selected'; } ?> value="false">False</option>
					<option <?php if(get_option('spo_show_excerpt') == 'true') { echo 'selected'; } ?> value="true">True</option>
					</select>
				</td> 
			</tr>
			<tr>
				<td><label for="spo_excerpt_length"><?php _e('Maximum length of excerpt (No. of characters):') ?></label></td>
				<td><input name="spo_excerpt_length" type="text" id="spo_excerpt_length" value="<?php echo get_option('spo_excerpt_length'); ?>" size="2" /> 
			</tr>
			<tr>
				<td><?php _e('Cut back excerpt to full word or sentence?') ?></td>
          		<td>
        			<select name="spo_excerpt_format" id="spo_excerpt_format">
        	  		<option <?php if(get_option('spo_excerpt_format') == 'char') { echo 'selected'; } ?> value="char">Do not trim</option>
					<option <?php if(get_option('spo_excerpt_format') == 'word') { echo 'selected'; } ?> value="word">Trim to full word</option>
					<option <?php if(get_option('spo_excerpt_format') == 'sent') { echo 'selected'; } ?> value="sent">Trim to sentence</option>
					</select>
				</td> 
			</tr>
			<tr>
				<td><?php _e('Add " ..." to excerpt?') ?></td>
          		<td>
        			<select name="spo_ellipsis" id="spo_ellipsis">
        	  		<option <?php if(get_option('spo_ellipsis') == 'false') { echo 'selected'; } ?> value="false">False</option>
					<option <?php if(get_option('spo_ellipsis') == 'true') { echo 'selected'; } ?> value="true">True</option>
					</select>
				</td> 
			</tr>
			<tr>
				<td><label for="spo_before_excerpt"><?php _e('Before') ?></label> / <label for="spo_after_excerpt"><?php _e('After (Excerpt)') ?></label>:</td>
				<td><input name="spo_before_excerpt" type="text" id="spo_before_excerpt" value="<?php echo htmlspecialchars(stripslashes(get_option('spo_before_excerpt'))); ?>" size="10" /> / <input name="spo_after_excerpt" type="text" id="spo_after_excerpt" value="<?php echo htmlspecialchars(stripslashes(get_option('spo_after_excerpt'))); ?>" size="10" /><em><small> <?php _e('For example: ') ?>&lt;li&gt;&lt;/li&gt; <?php _e('or') ?> &lt;dd&gt;&lt;/dd&gt;</small></em>
				</td>
			</tr>
			<tr>
				<td><label for="spo_show_private"><?php _e('Show password protected posts?') ?></label></td>
				<td>
            	<select name="spo_show_private" id="spo_show_private">
              	<option <?php if(get_option('spo_show_private') == 'false') { echo 'selected'; } ?> value="false">False</option>
              	<option <?php if(get_option('spo_show_private') == 'true') { echo 'selected'; } ?> value="true">True</option>
            	</select> 
				</td>
			</tr>
			<tr>
				<td><?php _e('Show static posts (i.e., pages) ?') ?></td>
          		<td>
        			<select name="spo_show_static" id="spo_show_static">
        	  		<option <?php if(get_option('spo_show_static') == 'false') { echo 'selected'; } ?> value="false">False</option>
					<option <?php if(get_option('spo_show_static') == 'true') { echo 'selected'; } ?> value="true">True</option>
					</select>
				</td> 
			</tr>
			<tr>
				<td><label for="spo_none_text"><?php _e('What to display if no posts can be found ? It could, e.g., be a simple message or a hyperlink to a favourite post.') ?></label>:</td>
				<td><input name="spo_none_text" type="text" id="spo_none_text" value="<?php echo htmlspecialchars(stripslashes(get_option('spo_none_text'))); ?>" size="40" /></td>
			</tr>
			<tr>
				<td><?php _e('Which authors, if any, do you want to exclude ?') ?></td>
          		<td>
					<table>	
					<?php 
						$excluded = explode(',', get_option('spo_excluded_authors'));
						$users = $wpdb->get_results("SELECT ID, user_login FROM $wpdb->users ORDER BY user_login");
						if ($users) {
							foreach ($users as $user) {
									if (false === in_array($user->ID, $excluded)) {
										$ischecked = '';
									} else {
										$ischecked = 'checked';
									}
									echo "\n\t<tr><td>$user->user_login</td><td><input type=\"checkbox\" name=\"spo_excluded_authors[]\" value=\"$user->ID\" $ischecked></td></tr>";
							}
						}	
					?>
					</table>
				</td> 
			</tr>
			<tr>
				<td><?php _e('Which categories, if any, do you want to exclude (bear in mind that if a post has multiple categories any category that is not excluded will override one that is):') ?></td>
          		<td>
					<table>	
					<?php 
						$excluded = explode(',', get_option('spo_excluded_cats'));
						display_cats($excluded, 'spo_excluded_cats'); 
					?>
					</table>
				</td> 
			</tr>
		</table>
		</fieldset>
		<p><div class="submit"><input type="submit" name="update_options" value="<?php _e('Update', 'update_rp') ?>"  style="font-weight:bold;" /></div></p>
		</form>    		
	</div>
		
    <div class="wrap"> 
		<h2><?php _e('Similar Posts Custom Fields') ?></h2>
		<form method="post">		
		<p><?php _e('By default this plugin processes every post each time it is viewed to find the terms to match. 
		This adds an avoidable overhead which can be bypassed if the pre-processed terms are stored
		in a custom field (called "similarterms"). This is done automatically whenever a post is published or edited
		but you can also opt to pre-process every post at once by pressing the <strong>Process</strong> 
		button below. It may take half a second per post. It will not overwrite any values that have 
		already been stored.') ?></p>
		<p><div class="submit">	
		<input type="submit" name="process_all" value="<?php _e('Process') ?>" style="font-weight:bold;" />
		</div></p>		
		<p><?php _e('There is also the option to clear <em>all</em> Similar Posts\' custom fields by pressing 
		the <strong>Clear</strong> button below.') ?></p>
		<p><div class="submit">
		<input type="submit" name="clear_all" value="<?php _e('Clear') ?>" style="font-weight:bold;" />
		</div></p> 		
		<p><?php _e('If you have been using the Related Posts plugin and have created keywords to
		sharpen up your matches you can import all your keywords into Similar Posts. This 
		overwrites the corresponding Similar Posts custom fields. To begin the import press the <strong>Import</strong> 
		button below.') ?></p>
		<p><div class="submit">
		<input type="submit" name="import_all" value="<?php _e('Import') ?>" style="font-weight:bold;" />
		</div></p>       
		</form>       
    </div>
	<?php
}

/*
	This very untidy function is called recursively from the options page above to display an indented table of categories and checkboxes
	If it didn't have to call itself it would be embedded in the page
	$excluded is the array of categories to exclude 
*/
function display_cats($excluded, $spo_name, $currentcat = 0, $currentparent = 0, $parent = 0, $level = 0, $categories = 0) {
	global $wpdb, $bgcolor;
	if (!$categories) {
		$categories = $wpdb->get_results("SELECT * FROM $wpdb->categories ORDER BY cat_name");
	}
	if ($categories) {
		foreach ($categories as $category) {
			if ($currentcat != $category->cat_ID && $parent == $category->category_parent) {
				$count = $wpdb->get_var("SELECT COUNT(post_id) FROM $wpdb->post2cat WHERE category_id = $category->cat_ID");
				$pad = str_repeat('&#8211; ', $level);
				$category->cat_name = wp_specialchars($category->cat_name);
				if (false === in_array($category->cat_ID, $excluded)) {
					$ischecked = '';
				} else {
					$ischecked = 'checked';
				}
				echo "\n\t<tr><td>$pad$category->cat_name</td><td><input type=\"checkbox\" name=\"".$spo_name."[]\" value=\"$category->cat_ID\" $ischecked></td></tr>";
				display_cats($excluded, $spo_name, $currentcat, $currentparent, $category->cat_ID, $level +1, $categories);
			}
		}
	} else {
		return false;
	}
}
	
/*
	Install the options page
*/
function similar_posts_option_menu() {
	if (function_exists('add_options_page')) {
		add_options_page(__('Similar Posts Options'), __('Similar Posts'), 1, __FILE__, 'similar_posts_options_page');
	}
}

function ensure_fulltext_index() {
	global $wpdb, $table_prefix;
	//check to see if the index is already in place
	$spo_index_exists = FALSE;
	$index_info = $wpdb->get_results('SHOW INDEX FROM ' . $table_prefix. 'posts', ARRAY_A);
	if (!$index_info) return FALSE;
	foreach ($index_info as $index) {
		if ($index['Key_name'] == 'post_similar') {
			$spo_index_exists = TRUE;
		}
	}
	if (!$spo_index_exists) {
		//try and set up the new index
		$result = $wpdb->query('ALTER TABLE `'.$table_prefix.'posts` ADD FULLTEXT `post_similar` ( `post_title` ,' . ' `post_content` )');
		if ($result === FALSE) {
			$spo_index_exists = FALSE;
		}
	}
	return $spo_index_exists;
}

/*
	This is the body of the plugin which gets 
*/

if (ensure_fulltext_index()) {
	$spo_index_exists = TRUE;
	//the next lines find the language WordPress is using
	$language = WPLANG;
	//if no language is specified make it the default which is 'en'
	if ($language == '') {
		$language = 'en';
	}
	$overusedwordsfile = $language.'.words.php';
	//see if there is a file of overused words in that language in the right directory
	if (!file_exists($overusedwordsfile)) {
		//if not revert to the default file
		$overusedwordsfile = 'en.words.php';
	}
	//the file contians a single php command to create the $overusedwords array -- see 'en.words.php' for an example
	require_once($overusedwordsfile);
	//set the default values of the options
	add_option('spo_limit', 5);
	add_option('spo_skip', 0);
	add_option('spo_excerpt_length', 50);
	add_option('spo_before_title', '<li>');
	add_option('spo_after_title', '</li>');
	add_option('spo_trim_before', 'false');
	add_option('spo_before_excerpt', '<li>');
	add_option('spo_after_excerpt', '</li>');
	add_option('spo_show_private', 'false');
	add_option('spo_show_excerpt', 'false');
	add_option('spo_show_static', 'false');
	add_option('spo_none_text', __('No Similar Posts'));
	// '9999' coressponds to no categories excluded
	add_option('spo_excluded_cats', '9999');
	add_option('spo_excluded_authors', '9999');
	add_option('spo_excerpt_format', 'char');
	add_option('spo_ellipsis', 'false');
	//install the term-similar actions into WordPress
	add_action('delete_post', 'delete_similar_terms', 1);
	add_action('edit_post', 'save_similar_terms', 1);
	add_action('publish_post', 'save_similar_terms', 1);
	add_action('admin_menu', 'similar_posts_option_menu');
} else {
	$spo_index_exists = FALSE;
 	_e("Problem creating the full text index for Similar Posts. Please check the instructions on how to create the index manually.");
}

?>